<?php
// core/ai_rewriter.php V4
// Gemini 2.0 Flash friendly: judul & isi diproses terpisah (Bahasa Indonesia gaya netral profesional)

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/helper.php';

/**
 * Panggil Gemini sekali (helper generik).
 */
function gemini_call_text($apiKey, $model, $prompt, $temperature, $maxTokens, $topP, $topK) {
    $endpoint = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$apiKey}";

    $payload = [
        'contents' => [
            [
                'parts' => [
                    ['text' => $prompt]
                ]
            ]
        ],
        'generationConfig' => [
            'temperature'     => floatval($temperature),
            'maxOutputTokens' => intval($maxTokens),
            'topP'            => floatval($topP),
            'topK'            => intval($topK),
        ]
    ];

    $headers = ['Content-Type: application/json'];

    $response = http_post($endpoint, json_encode($payload), $headers);
    $json     = json_decode($response, true);

    if (!isset($json['candidates'][0]['content']['parts'][0]['text'])) {
        throw new Exception('Response Gemini tidak sesuai format. Respon mentah: ' . $response);
    }

    return trim($json['candidates'][0]['content']['parts'][0]['text']);
}

/**
 * Rewrite isi artikel ke Bahasa Indonesia (tanpa judul).
 */
function rewrite_body_id($template, $originalTitle, $originalContent) {
    $apiKey = $template['api_key'];
    $model  = $template['model'];

    if (!$apiKey || !$model) {
        throw new Exception('API Key atau Model Gemini belum di-set di Template.');
    }

    $prompt = "You are a professional football news writer.

"
        . "Rewrite the following news article into Bahasa Indonesia with a neutral and professional journalistic tone, "
        . "like a wire news agency. Use clear and concise sentences, explain context (who, what, when, where, why, how), "
        . "and keep all important facts.

"
        . "Rules:
"
        . "- Language: Bahasa Indonesia only.
"
        . "- Minimum length: 400 words.
"
        . "- Use short paragraphs.
"
        . "- Add some H2 and H3 subheadings where relevant (use markdown style ## and ###).
"
        . "- Remove any text like 'Isi Artikel:' if present.
"
        . "- If the source contains 'dailystar.co.uk', replace it with 'nextsport.id'.
"
        . "- The FIRST paragraph MUST start exactly with:
"
        . "  <strong>NextSport Transfers - </strong>
"
        . "- DO NOT write any title. Output ONLY the article body.

"
        . "Source title:
{$originalTitle}

"
        . "Source article:
{$originalContent}

"
        . "Output only the rewritten article body in Bahasa Indonesia.";

    $text = gemini_call_text(
        $apiKey,
        $model,
        $prompt,
        $template['temperature'],
        $template['max_tokens'],
        $template['top_p'],
        $template['top_k']
    );

    // Bersihkan markdown ** kalau ada
    $text = str_replace('**', '', $text);

    // Pastikan ada prefix <strong>NextSport Transfers - </strong>
    $prefix = '<strong>NextSport Transfers - </strong>';
    if (stripos($text, $prefix) !== 0) {
        $text = $prefix . ' ' . ltrim($text);
    }

    // Konversi markdown heading menjadi HTML
    $text = preg_replace('/^###\s*(.+)$/m', '<h3>$1</h3>', $text);
    $text = preg_replace('/^##\s*(.+)$/m', '<h2>$1</h2>', $text);

    return trim($text);
}

/**
 * Buat judul Bahasa Indonesia.
 */
function rewrite_title_id($template, $originalTitle) {
    $apiKey = $template['api_key'];
    $model  = $template['model'];

    if (!$apiKey || !$model) {
        throw new Exception('API Key atau Model Gemini belum di-set di Template.');
    }

    $prompt = "You are a professional sports news headline writer.

"
        . "Create a new headline in Bahasa Indonesia based on the following English headline.

"
        . "Rules:
"
        . "- Maximum 50 characters.
"
        . "- No punctuation at all (no dot, comma, question mark, exclamation mark, dash, quotes).
"
        . "- Do not mention any domain name.
"
        . "- Slightly catchy but still neutral and professional.
"
        . "- Output ONLY the headline text, nothing else.

"
        . "Original headline:
{$originalTitle}";

    $title = gemini_call_text(
        $apiKey,
        $model,
        $prompt,
        $template['temperature'],
        128,
        $template['top_p'],
        $template['top_k']
    );

    // Hilangkan "| OneFootball" kalau masih terbawa
    $title = preg_replace('/\|\s*OneFootball/i', '', $title);
    $title = trim($title);

    return $title;
}

/**
 * Wrapper: rewrite judul + isi.
 */
function rewrite_with_gemini($template, $originalTitle, $originalContent) {
    $newBody  = rewrite_body_id($template, $originalTitle, $originalContent);
    $newTitle = rewrite_title_id($template, $originalTitle);

    if ($newTitle === '') {
        // Fallback: pakai judul asli tanpa "| OneFootball"
        $newTitle = preg_replace('/\|\s*OneFootball/i', '', $originalTitle);
        $newTitle = trim($newTitle);
    }

    return [
        'title'   => $newTitle,
        'content' => $newBody,
    ];
}

/**
 * Proses batch artikel: ambil yang status = 'draft', rewrite, lalu simpan jadi 'processed'.
 */
function process_rewrite_batch($templateId, $limit = 3) {
    global $pdo;

    $stmtTemplate = $pdo->prepare("SELECT * FROM templates WHERE id = ?");
    $stmtTemplate->execute([$templateId]);
    $template = $stmtTemplate->fetch();
    if (!$template) {
        throw new Exception('Template tidak ditemukan (ID: ' . $templateId . ').');
    }

    $stmt = $pdo->prepare("SELECT * FROM articles WHERE status = 'draft' ORDER BY id ASC LIMIT ?");
    $stmt->bindValue(1, (int)$limit, PDO::PARAM_INT);
    $stmt->execute();
    $articles = $stmt->fetchAll();

    foreach ($articles as $article) {
        try {
            echo "Rewriting article ID {$article['id']}...\n";
            $rew = rewrite_with_gemini($template, $article['title'], $article['content']);

            $upd = $pdo->prepare("UPDATE articles SET title = ?, rewritten = ?, status = 'processed' WHERE id = ?");
            $upd->execute([$rew['title'], $rew['content'], $article['id']]);

            echo "OK: ID {$article['id']} berhasil di-rewrite.\n";
        } catch (Exception $e) {
            echo "GAGAL rewrite ID {$article['id']}: " . $e->getMessage() . "\n";
        }
    }
}
